<?php
class PerdiemController extends Controller
{
    public function index()
    {
        $db = new Database();
        
        // Get all perdiem records with joined data
        $result = $db->selectQuery("
            SELECT 
                p.id,
                p.client_id,
                c.short_name as client_code,
                p.transport_mode_id,
                t.transport_mode_name,
                p.goods_type_id,
                g.goods_type,
                p.location_id,
                l.main_location_name,
                p.perdiem_amount,
                p.im_ex_lo_id,
                p.display,
                p.created_at,
                p.updated_at
            FROM perdiem_master_t p
            LEFT JOIN clients_t c ON p.client_id = c.id
            LEFT JOIN transport_mode_master_t t ON p.transport_mode_id = t.id
            LEFT JOIN type_of_goods_master_t g ON p.goods_type_id = g.id
            LEFT JOIN main_office_master_t l ON p.location_id = l.id
            ORDER BY p.id DESC
        ");

        // Get clients for dropdown
        $clients = $db->selectData('clients_t', 'id, short_name, company_name', ['display' => 'Y'], 'short_name ASC');

        // Get transport modes for dropdown
        $transportModes = $db->selectData('transport_mode_master_t', 'id, transport_mode_name', ['display' => 'Y'], 'transport_mode_name ASC');

        // Get goods types for dropdown
        $goodsTypes = $db->selectData('type_of_goods_master_t', 'id, goods_type', ['display' => 'Y'], 'goods_type ASC');

        // Get locations for dropdown
        $locations = $db->selectData('main_office_master_t', 'id, main_location_name', ['display' => 'Y'], 'main_location_name ASC');

        $data = [
            'title' => 'Per Diem Master',
            'result' => $result ?: [],
            'clients' => $clients ?: [],
            'transportModes' => $transportModes ?: [],
            'goodsTypes' => $goodsTypes ?: [],
            'locations' => $locations ?: []
        ];

        $this->viewWithLayout('masters/perdiem', $data);
    }

    public function crudData($action = 'insertion')
    {
        $db = new Database();
        $table = 'perdiem_master_t';

        function sanitize($value)
        {
            return htmlspecialchars(trim($value), ENT_QUOTES, 'UTF-8');
        }

        // BULK INSERT FOR MULTIPLE CLIENTS
        if ($action === 'insertion' && $_SERVER['REQUEST_METHOD'] === 'POST') {
            header('Content-Type: application/json');
            
            // Get multiple client IDs
            $client_ids = isset($_POST['client_id']) ? $_POST['client_id'] : [];
            
            // If single value, convert to array
            if (!is_array($client_ids)) {
                $client_ids = [$client_ids];
            }
            
            $transport_mode_id = (int)($_POST['transport_mode_id'] ?? 0);
            $goods_type_id = (int)($_POST['goods_type_id'] ?? 0);
            $im_ex_lo_id = (int)($_POST['im_ex_lo_id'] ?? 0);

            $location_id = (int)($_POST['location_id'] ?? 0);
            $perdiem_amount = floatval($_POST['perdiem_amount'] ?? 0);
            $display = isset($_POST['display']) && in_array($_POST['display'], ['Y', 'N']) ? $_POST['display'] : 'Y';

            // Validation
            if (empty($client_ids) || $transport_mode_id <= 0 || $im_ex_lo_id<=0 ||$goods_type_id <= 0 || $location_id <= 0) {
                echo json_encode(['success' => false, 'message' => 'All fields are required.']);
                exit;
            }

            if ($perdiem_amount <= 0) {
                echo json_encode(['success' => false, 'message' => 'Per diem amount must be greater than zero.']);
                exit;
            }

            $successCount = 0;
            $skipCount = 0;
            $failCount = 0;
            $skippedClients = [];
            
            // Loop through each selected client
            foreach ($client_ids as $client_id) {
                $client_id = (int)$client_id;
                
                if ($client_id <= 0) {
                    $failCount++;
                    continue;
                }

                // Check if combination already exists
                $existCheck = $db->selectData($table, 'id', [
                    'client_id' => $client_id,
                    'transport_mode_id' => $transport_mode_id,
                    'goods_type_id' => $goods_type_id,
                    'location_id' => $location_id,
                    'im_ex_lo_id' =>  $im_ex_lo_id
                ]);

                if (!empty($existCheck)) {
                    $skipCount++;
                    // Get client name for skipped message
                    $clientInfo = $db->selectData('clients_t', 'short_name', ['id' => $client_id]);
                    if (!empty($clientInfo)) {
                        $skippedClients[] = $clientInfo[0]['short_name'];
                    }
                    continue;
                }

                // Insert per diem record for this client
                $data = [
                    'client_id' => $client_id,
                    'transport_mode_id' => $transport_mode_id,
                    'goods_type_id' => $goods_type_id,
                    'location_id' => $location_id,
                    'perdiem_amount' => $perdiem_amount,
                    'im_ex_lo_id' =>  $im_ex_lo_id,
                    'display' => $display,
                    'created_by' => 1,
                    'updated_by' => 1,
                ];

                $insertId = $db->insertData($table, $data);
                
                if ($insertId) {
                    $successCount++;
                } else {
                    $failCount++;
                }
            }

            // Build response message
            $message = [];
            if ($successCount > 0) {
                $message[] = "✅ {$successCount} per diem record(s) added successfully";
            }
            if ($skipCount > 0) {
                $message[] = "⚠️ {$skipCount} duplicate(s) skipped: " . implode(', ', $skippedClients);
            }
            if ($failCount > 0) {
                $message[] = "❌ {$failCount} record(s) failed";
            }

            echo json_encode([
                'success' => $successCount > 0,
                'message' => implode('. ', $message),
                'details' => [
                    'success' => $successCount,
                    'skipped' => $skipCount,
                    'failed' => $failCount
                ]
            ]);
            exit;
        }

        // UPDATE
        if ($action === 'updation' && $_SERVER['REQUEST_METHOD'] === 'POST') {
            header('Content-Type: application/json');
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            
            if ($id <= 0) {
                echo json_encode(['success' => false, 'message' => 'Invalid per diem ID.']);
                exit;
            }

            $client_id = (int)($_POST['client_id'] ?? 0);
            $transport_mode_id = (int)($_POST['transport_mode_id'] ?? 0);
            $goods_type_id = (int)($_POST['goods_type_id'] ?? 0);
            $location_id = (int)($_POST['location_id'] ?? 0);
             $im_ex_lo_id =($_POST['im_ex_lo_id'] ?? 0);
            $perdiem_amount = floatval($_POST['perdiem_amount'] ?? 0);
            $display = isset($_POST['display']) && in_array($_POST['display'], ['Y', 'N']) ? $_POST['display'] : 'Y';

            if ($client_id <= 0 || $transport_mode_id <= 0 || $im_ex_lo_id <= 0|| $goods_type_id <= 0 || $location_id <= 0) {
                echo json_encode(['success' => false, 'message' => 'All fields are required.']);
                exit;
            }

            if ($perdiem_amount <= 0) {
                echo json_encode(['success' => false, 'message' => 'Per diem amount must be greater than zero.']);
                exit;
            }

            // Check for duplicate (excluding current record)
           $sql = "
    SELECT id FROM perdiem_master_t
    WHERE client_id = $client_id
    AND transport_mode_id = $transport_mode_id
    AND goods_type_id = $goods_type_id
    AND location_id = $location_id
    AND im_ex_lo_id = $im_ex_lo_id
    AND id != $id
";

$existCheck = $db->selectQuery($sql);


            if (!empty($existCheck)) {
                echo json_encode(['success' => false, 'message' => 'This combination already exists.']);
                exit;
            }

            $data = [
                'client_id' => $client_id,
                'transport_mode_id' => $transport_mode_id,
                'goods_type_id' => $goods_type_id,
                'location_id' => $location_id,
                'im_ex_lo_id' => $im_ex_lo_id,
                'perdiem_amount' => $perdiem_amount,
                'display' => $display,
                'updated_by' => 1,
            ];

            $update = $db->updateData($table, $data, ['id' => $id]);
            echo json_encode([
                'success' => $update ? true : false,
                'message' => $update ? 'Per diem updated successfully!' : 'Update failed.'
            ]);
            exit;
        }

        // DELETE
        if ($action === 'deletion') {
            header('Content-Type: application/json');
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            
            if ($id <= 0) {
                echo json_encode(['success' => false, 'message' => 'Invalid ID for deletion.']);
                exit;
            }

            $delete = $db->deleteData($table, ['id' => $id]);
            echo json_encode([
                'success' => $delete ? true : false,
                'message' => $delete ? 'Per diem deleted successfully!' : 'Delete failed.'
            ]);
            exit;
        }

        echo json_encode(['success' => false, 'message' => 'Invalid request.']);
        exit;
    }

    public function getPerdiemById()
    {
        header('Content-Type: application/json');
        $db = new Database();
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

        if ($id <= 0) {
            echo json_encode(['success' => false, 'message' => 'Invalid ID.']);
            exit;
        }

        $perdiem = $db->selectData('perdiem_master_t', '*', ['id' => $id]);

        echo json_encode(!empty($perdiem)
            ? ['success' => true, 'data' => $perdiem[0]]
            : ['success' => false, 'message' => 'Per diem not found.']
        );
        exit;
    }
}
?>