<?php

class PaymentController extends Controller
{
    /* -----------------------------------------------------------
       LOAD PAGE
    -------------------------------------------------------------*/
    public function index()
    {
        $db = new Database();
        $dept     = $db->selectData('department_master_t','*',[]);
        $loc      = $db->selectData('main_office_master_t','*',[]);
        $client   = $db->selectData('clients_t','*',[]);
        $currency = $db->selectData('currency_master_t','*',[],2);
        $expense  = $db->selectData('expense_type_master_t','*',[]);

        $data = [
            'title'     => 'Payment Request',
            'dept'      => $dept,
            'loc'       => $loc,
            'client'    => $client,
            'currency'  => $currency,
            'expense'   => $expense
        ];

        $this->viewWithLayout('payment/payment', $data);
    }

    /* -----------------------------------------------------------
       GET STATUS COUNTS FOR FILTER CARDS
    -------------------------------------------------------------*/
    public function get_status_counts()
    {
        header('Content-Type: application/json');
        
        $db = new Database();
        
        try {
            // Total count
            $totalSql = "SELECT COUNT(*) as count FROM payment_requests";
            $totalResult = $db->customQuery($totalSql);
            $total = $totalResult[0]['count'] ?? 0;
            
            // Waiting Department Approval (no rejections)
            $waitingDeptSql = "SELECT COUNT(*) as count 
                FROM payment_requests 
                WHERE dept_approval IS NULL 
                AND COALESCE(dept_approval, 0) != -1
                AND COALESCE(finance_approval, 0) != -1
                AND COALESCE(management_approval, 0) != -1
                AND COALESCE(paid_approval, 0) != -1";
            $waitingDeptResult = $db->customQuery($waitingDeptSql);
            $waiting_dept = $waitingDeptResult[0]['count'] ?? 0;
            
            // Waiting Finance Approval (dept approved, no rejections)
            $waitingFinanceSql = "SELECT COUNT(*) as count 
                FROM payment_requests 
                WHERE dept_approval = 1 
                AND finance_approval IS NULL
                AND COALESCE(finance_approval, 0) != -1
                AND COALESCE(management_approval, 0) != -1
                AND COALESCE(paid_approval, 0) != -1";
            $waitingFinanceResult = $db->customQuery($waitingFinanceSql);
            $waiting_finance = $waitingFinanceResult[0]['count'] ?? 0;
            
            // Waiting Management Approval (dept & finance approved, no rejections)
            $waitingMgmtSql = "SELECT COUNT(*) as count 
                FROM payment_requests 
                WHERE dept_approval = 1 
                AND finance_approval = 1 
                AND management_approval IS NULL
                AND COALESCE(management_approval, 0) != -1
                AND COALESCE(paid_approval, 0) != -1";
            $waitingMgmtResult = $db->customQuery($waitingMgmtSql);
            $waiting_mgmt = $waitingMgmtResult[0]['count'] ?? 0;
            
            // Paid
            $paidSql = "SELECT COUNT(*) as count 
                FROM payment_requests 
                WHERE paid_approval = 1";
            $paidResult = $db->customQuery($paidSql);
            $paid = $paidResult[0]['count'] ?? 0;
            
            // Rejected (any stage)
            $rejectedSql = "SELECT COUNT(*) as count 
                FROM payment_requests 
                WHERE dept_approval = -1 
                OR finance_approval = -1 
                OR management_approval = -1 
                OR paid_approval = -1";
            $rejectedResult = $db->customQuery($rejectedSql);
            $rejected = $rejectedResult[0]['count'] ?? 0;
            
            echo json_encode([
                'success' => true,
                'data' => [
                    'total' => $total,
                    'waiting_dept' => $waiting_dept,
                    'waiting_finance' => $waiting_finance,
                    'waiting_mgmt' => $waiting_mgmt,
                    'paid' => $paid,
                    'rejected' => $rejected
                ]
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'message' => 'Error fetching status counts: ' . $e->getMessage()
            ]);
        }
        exit;
    }

    /* -----------------------------------------------------------
       DATATABLE LIST WITH STATUS FILTER - FIXED SEARCH
    -------------------------------------------------------------*/
    public function get_list()
    {
        header('Content-Type: application/json');

        $db = new Database();

        $draw   = intval($_POST['draw'] ?? 1);
        $start  = intval($_POST['start'] ?? 0);
        $length = intval($_POST['length'] ?? 10);
        $search = trim($_POST['search']['value'] ?? '');
        
        // GET STATUS FILTER
        $statusFilter = $_POST['status_filter'] ?? 'all';
            $userRoleId = $_SESSION['USER']['role_id'] ?? 0;

        // Base SQL with all necessary JOINs
        $sql = "SELECT pr.*, 
                       d.department_name, 
                       c.short_name AS client_name,
                       ex.expense_type_name
                FROM payment_requests pr
                LEFT JOIN department_master_t d ON d.id = pr.department
                LEFT JOIN clients_t c ON c.id = pr.client_id
                LEFT JOIN expense_type_master_t ex ON ex.id = pr.expense_type
                WHERE 1=1 ";

        $params = [];
        if ($userRoleId == 5) {
        $sql .= " AND pr.payment_type = 'Cash' ";
    }
        // APPLY STATUS FILTER
        switch ($statusFilter) {
            case 'waiting_dept':
                $sql .= " AND pr.dept_approval IS NULL 
                          AND COALESCE(pr.dept_approval, 0) != -1
                          AND COALESCE(pr.finance_approval, 0) != -1
                          AND COALESCE(pr.management_approval, 0) != -1
                          AND COALESCE(pr.paid_approval, 0) != -1";
                break;
                
            case 'waiting_finance':
                $sql .= " AND pr.dept_approval = 1 
                          AND pr.finance_approval IS NULL
                          AND COALESCE(pr.finance_approval, 0) != -1
                          AND COALESCE(pr.management_approval, 0) != -1
                          AND COALESCE(pr.paid_approval, 0) != -1";
                break;
                
            case 'waiting_mgmt':
                $sql .= " AND pr.dept_approval = 1 
                          AND pr.finance_approval = 1 
                          AND pr.management_approval IS NULL
                          AND COALESCE(pr.management_approval, 0) != -1
                          AND COALESCE(pr.paid_approval, 0) != -1";
                break;
                
            case 'paid':
                $sql .= " AND pr.paid_approval = 1";
                break;
                
            case 'rejected':
                $sql .= " AND (pr.dept_approval = -1 
                            OR pr.finance_approval = -1 
                            OR pr.management_approval = -1 
                            OR pr.paid_approval = -1)";
                break;
                
            // case 'all':
            // default:
            //     // No additional filter
            //     break;
        }

        // SEARCH FILTER - FIXED
        if (!empty($search)) {
            $sql .= " AND (
                pr.beneficiary LIKE :search 
                OR pr.motif LIKE :search 
                OR pr.requestee LIKE :search
                OR pr.cash_collector LIKE :search
                OR CAST(pr.id AS CHAR) LIKE :search
                OR CAST(pr.amount AS CHAR) LIKE :search
                OR d.department_name LIKE :search
                OR c.short_name LIKE :search
                OR ex.expense_type_name LIKE :search
                OR pr.payment_type LIKE :search
            )";
            $params[':search'] = "%{$search}%";
        }

        // Count filtered with same JOINs and filters
        $countSql = "SELECT COUNT(*) AS filtered 
                     FROM payment_requests pr 
                     LEFT JOIN department_master_t d ON d.id = pr.department
                     LEFT JOIN clients_t c ON c.id = pr.client_id
                     LEFT JOIN expense_type_master_t ex ON ex.id = pr.expense_type
                     WHERE 1=1 ";
        if ($userRoleId == 5) {
        $countSql .= " AND pr.payment_type = 'Cash' ";
    }
        // Apply same status filter to count
        switch ($statusFilter) {
            case 'waiting_dept':
                $countSql .= " AND pr.dept_approval IS NULL 
                               AND COALESCE(pr.dept_approval, 0) != -1
                               AND COALESCE(pr.finance_approval, 0) != -1
                               AND COALESCE(pr.management_approval, 0) != -1
                               AND COALESCE(pr.paid_approval, 0) != -1";
                break;
            case 'waiting_finance':
                $countSql .= " AND pr.dept_approval = 1 
                               AND pr.finance_approval IS NULL
                               AND COALESCE(pr.finance_approval, 0) != -1
                               AND COALESCE(pr.management_approval, 0) != -1
                               AND COALESCE(pr.paid_approval, 0) != -1";
                break;
            case 'waiting_mgmt':
                $countSql .= " AND pr.dept_approval = 1 
                               AND pr.finance_approval = 1 
                               AND pr.management_approval IS NULL
                               AND COALESCE(pr.management_approval, 0) != -1
                               AND COALESCE(pr.paid_approval, 0) != -1";
                break;
            case 'paid':
                $countSql .= " AND pr.paid_approval = 1";
                break;
            case 'rejected':
                $countSql .= " AND (pr.dept_approval = -1 
                                OR pr.finance_approval = -1 
                                OR pr.management_approval = -1 
                                OR pr.paid_approval = -1)";
                break;
        }

        if (!empty($search)) {
            $countSql .= " AND (
                pr.beneficiary LIKE :search 
                OR pr.motif LIKE :search 
                OR pr.requestee LIKE :search
                OR pr.cash_collector LIKE :search
                OR CAST(pr.id AS CHAR) LIKE :search
                OR CAST(pr.amount AS CHAR) LIKE :search
                OR d.department_name LIKE :search
                OR c.short_name LIKE :search
                OR ex.expense_type_name LIKE :search
                OR pr.payment_type LIKE :search
            )";
        }

        $countResult = $db->customQuery($countSql, $params);
        $recordsFiltered = $countResult[0]['filtered'] ?? 0;

        // Count total
        $totalSql = "SELECT COUNT(*) AS total FROM payment_requests";
        $totalResult = $db->customQuery($totalSql);
        $recordsTotal = $totalResult[0]['total'] ?? 0;

        // Pagination
        $sql .= " ORDER BY pr.id DESC LIMIT :start, :length";
        $params[':start'] = $start;
        $params[':length'] = $length;

        $data = $db->customQuery($sql, $params);

        echo json_encode([
            "draw" => $draw,
            "recordsTotal" => $recordsTotal,
            "recordsFiltered" => $recordsFiltered,
            "data" => $data
        ]);
        exit;
    }

    /* -----------------------------------------------------------
       CHECK MCA REFERENCE DUPLICATE
    -------------------------------------------------------------*/
    public function check_mca_duplicate()
    {
        header('Content-Type: application/json');
        
        $mca_ref = $_POST['mca_ref'] ?? '';
        $expense_type = $_POST['expense_type'] ?? '';
        $current_payment_id = $_POST['payment_id'] ?? null;
        
        if (empty($mca_ref) || empty($expense_type)) {
            echo json_encode(['exists' => false]);
            exit;
        }
        
        $db = new Database();
        
        // Check if this MCA ref + expense type combo exists
        $sql = "SELECT pr.id, pr.mca_data 
                FROM payment_requests pr 
                WHERE pr.expense_type = :expense_type";
        
        if ($current_payment_id) {
            $sql .= " AND pr.id != :payment_id";
        }
        
        $params = [':expense_type' => $expense_type];
        if ($current_payment_id) {
            $params[':payment_id'] = $current_payment_id;
        }
        
        $results = $db->customQuery($sql, $params);
        
        foreach ($results as $row) {
            if (!empty($row['mca_data'])) {
                $mcaData = json_decode($row['mca_data'], true);
                if ($mcaData) {
                    foreach ($mcaData as $mca) {
                        if (isset($mca['mca_ref']) && $mca['mca_ref'] === $mca_ref) {
                            echo json_encode([
                                'exists' => true,
                                'payment_id' => $row['id'],
                                'message' => "MCA Reference '{$mca_ref}' is already used for this Expense Type in Payment Request #{$row['id']}"
                            ]);
                            exit;
                        }
                    }
                }
            }
        }
        
        echo json_encode(['exists' => false]);
        exit;
    }

    /* -----------------------------------------------------------
       STORE PAYMENT REQUEST  (INSERT / UPDATE)
    -------------------------------------------------------------*/
    public function store()
    {
        header('Content-Type: application/json');
        
        $db = new Database();

        $action        = $_POST['action'] ?? 'insert';
        $payment_id    = $_POST['payment_id'] ?? null;

        $department_id = $_POST['department'] ?? null; 
        $location_id   = $_POST['location'] ?? null;
        $beneficiary   = $_POST['beneficiary'] ?? null;
        $requestee     = $_POST['requestee'] ?? null;

        $client_id     = $_POST['client_id'] ?? null;
        $pay_for       = $_POST['pay_for'] ?? null;
        $currency_id   = $_POST['currency'] ?? null;
        $amount        = $_POST['amount'] ?? 0;
        $payment_type  = $_POST['payment_type'] ?? null;
        $expense_type  = $_POST['expense_type'] ?? null;
        $motif         = $_POST['motif'] ?? null;
        $cash_collector = $_POST['cash_collector'] ?? null;
        
        $mca_refs      = $_POST['mca_reference'] ?? [];
        $mca_amounts   = $_POST['mca_amount'] ?? [];

        // Basic validation - INCLUDING CLIENT
        if (!$department_id || !$location_id || !$client_id || !$currency_id || !$amount || !$expense_type) {
            echo json_encode(['success' => false, 'message' => 'All required fields must be filled (Department, Location, Client, Currency, Amount, Expense Type)']);
            exit;
        }

        // MCA total validation
        if (!empty($mca_amounts)) {
            $totalMca = array_sum(array_map('floatval', $mca_amounts));

            if (abs((float)$totalMca - (float)$amount) > 0.01) {
                echo json_encode([
                    'success' => false,
                    'message' => "MCA total ($totalMca) must match Amount ($amount)"
                ]);
                exit;
            }
        }
         
        // FILE UPLOAD HANDLING
        $upload_dir = __DIR__ . "/../../public/uploads/payments/";

        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }

        $file_paths = [];

        foreach (['file1', 'file2'] as $fileKey) {

            if (!empty($_FILES[$fileKey]['name'])) {

                $fileTmp  = $_FILES[$fileKey]['tmp_name'];
                $fileName = time() . "_" . basename($_FILES[$fileKey]['name']);
                $fileDest = $upload_dir . $fileName;

                if (move_uploaded_file($fileTmp, $fileDest)) {
                    $file_paths[$fileKey] = "uploads/payments/" . $fileName;
                } else {
                    echo json_encode(['success' => false, 'message' => "Failed to upload file: " . $fileKey]);
                    exit;
                }
            }
        }

        $db->beginTransaction();

        try {

            // Build MCA JSON array
            $mca_data = [];
            foreach ($mca_refs as $i => $ref) {
                if (!trim($ref) || !isset($mca_amounts[$i])) continue;

                $mca_data[] = [
                    "mca_ref" => trim($ref),
                    "amount"  => (float)$mca_amounts[$i]
                ];
            }

            // Insert/update data
            $paymentData = [
                "beneficiary"  => $beneficiary,
                "requestee"    => $requestee,
                "department"   => $department_id,
                "location_id"  => $location_id,
                "client_id"    => $client_id,
                "pay_for"      => $pay_for,
                "currency"     => $currency_id,
                "amount"       => $amount,
                "payment_type" => $payment_type,
                "expense_type" => $expense_type,
                "motif"        => $motif,
                "cash_collector" => $cash_collector,

                // Save first MCA reference + whole MCA JSON
                "mca_ref"      => $mca_data[0]['mca_ref'] ?? null,
                "mca_data"     => json_encode($mca_data),
            ];

            // Only update file paths if new files uploaded
            if (isset($file_paths['file1'])) {
                $paymentData['file1_path'] = $file_paths['file1'];
            }
            if (isset($file_paths['file2'])) {
                $paymentData['file2_path'] = $file_paths['file2'];
            }

            if ($action === "update" && $payment_id) {

                $paymentData['updated_at'] = date('Y-m-d H:i:s');
                $paymentData['updated_by'] = $_SESSION['userid'] ?? 0;
                
                // Check if this is a resubmit after rejection
                $existingSql = "SELECT * FROM payment_requests WHERE id = :id";
                $existingResult = $db->customQuery($existingSql, [':id' => $payment_id]);
                $existingPayment = $existingResult[0] ?? null;
                
                if ($existingPayment && (
                    $existingPayment['dept_approval'] == -1 ||
                    $existingPayment['finance_approval'] == -1 ||
                    $existingPayment['management_approval'] == -1 ||
                    $existingPayment['paid_approval'] == -1
                )) {
                    // RESET APPROVALS WHEN EDITING AFTER REJECTION
                    $paymentData['dept_approval'] = null;
                    $paymentData['dept_approved_at'] = null;
                    $paymentData['dept_approved_by'] = null;
                    $paymentData['dept_notes'] = null;
                    
                    $paymentData['finance_approval'] = null;
                    $paymentData['finance_approved_at'] = null;
                    $paymentData['finance_approved_by'] = null;
                    $paymentData['finance_notes'] = null;
                    
                    $paymentData['management_approval'] = null;
                    $paymentData['management_approved_at'] = null;
                    $paymentData['management_approved_by'] = null;
                    $paymentData['management_notes'] = null;
                    
                    $paymentData['paid_approval'] = null;
                    $paymentData['paid_approved_at'] = null;
                    $paymentData['paid_approved_by'] = null;
                    $paymentData['paid_notes'] = null;
                }

                $db->updateData('payment_requests', $paymentData, ['id' => $payment_id]);

            } else {

                $paymentData['created_at'] = date('Y-m-d H:i:s');
                $paymentData['created_by'] = $_SESSION['userid'] ?? 0;

                $payment_id = $db->insertData('payment_requests', $paymentData);
            }

            $db->commit();

            echo json_encode([
                'success' => true,
                'message' => ($action === 'update')
                    ? 'Payment Updated Successfully'
                    : 'Payment Created Successfully'
            ]);

        } catch (Exception $e) {

            $db->rollBack();
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit;
    }

    /* -----------------------------------------------------------
       GET SINGLE PAYMENT - WITH USER NAMES
    -------------------------------------------------------------*/
   public function get_single()
{
    header('Content-Type: application/json');

    $id = $_POST['id'] ?? null;

    if (empty($id) || !is_numeric($id)) {
        echo json_encode([
            "success" => false,
            "message" => "Invalid Payment ID"
        ]);
        exit;
    }

    $db = new Database();

    $sql = "SELECT pr.*, 
                   d.department_name, 
                   c.short_name AS client_name, 
                   cu.currency_short_name, 
                   of.main_location_name, 
                   ex.expense_type_name,
                   u1.full_name as dept_approved_by_name,
                   u2.full_name as finance_approved_by_name,
                   u3.full_name as management_approved_by_name,
                   u4.full_name as paid_approved_by_name
            FROM payment_requests pr
            LEFT JOIN department_master_t d ON d.id = pr.department
            LEFT JOIN clients_t c ON c.id = pr.client_id
            LEFT JOIN currency_master_t cu ON cu.id = pr.currency
            LEFT JOIN main_office_master_t of ON of.id = pr.location_id
            LEFT JOIN expense_type_master_t ex ON ex.id = pr.expense_type
            LEFT JOIN users_t u1 ON u1.id = pr.dept_approved_by
            LEFT JOIN users_t u2 ON u2.id = pr.finance_approved_by
            LEFT JOIN users_t u3 ON u3.id = pr.management_approved_by
            LEFT JOIN users_t u4 ON u4.id = pr.paid_approved_by
            WHERE pr.id = :id
            LIMIT 1";

    $payment = $db->customQuery($sql, [':id' => $id]);

    if (!$payment || count($payment) == 0) {
        echo json_encode(["success" => false, "message" => "Payment record not found"]);
        exit;
    }

    $row = $payment[0];

    // Decode MCA data
    $mcaData = [];
    if (!empty($row["mca_data"])) {
        $decoded = json_decode($row["mca_data"], true);
        if (json_last_error() === JSON_ERROR_NONE) {
            $mcaData = $decoded;
        }
    }

    echo json_encode([
        "success" => true,
        "data" => $row,
        "mca_refs" => $mcaData
    ]);
    exit;
}

    /* -----------------------------------------------------------
       DELETE
    -------------------------------------------------------------*/
    public function delete($id)
    {
        header('Content-Type: application/json');
        
        $db = new Database();

        $db->deleteData("payment_requests", ["id" => $id]);

        echo json_encode([
            "success" => true,
            "message" => "Payment Request Deleted Successfully"
        ]);
        exit;
    }

    /* -----------------------------------------------------------
       GET EXPENSE TYPES BY CATEGORY
    -------------------------------------------------------------*/
    public function getExpenseTypesByCategory()
    { 
        header('Content-Type: application/json');

        $category = $_POST['pay_for'] ?? null;
        
        // Column mapping
        $columns = [
            '0' => 'import',
            '1' => 'export',
            '2' => 'local',
            '3' => 'other',
            '4' => 'advance'
        ];

        if (!isset($columns[$category])) {
            echo json_encode([]);
            exit;
        }

        $column = $columns[$category];

        $db = new Database();

        // Fetch expense types where column = 1
        $expenseTypes = $db->selectData(
            "expense_type_master_t",
            "*",
            [$column => 1]
        );

        echo json_encode($expenseTypes);
        exit;
    }

    /* -----------------------------------------------------------
       UPDATE APPROVAL
    -------------------------------------------------------------*/
    public function update_approval()
    {
        header('Content-Type: application/json');

        $db     = new Database();
        $id     = $_POST['id'] ?? null;
        $action = $_POST['action'] ?? null;

        if (!$id || !$action) {
            echo json_encode(["success" => false, "message" => "Invalid Request"]);
            exit;
        }

        $updateData = [];
        $where = ["id" => $id];

        switch ($action) {

            case "dept_approve":
                $chargeback_needed = $_POST['chargeback_needed'] ?? 0;
                $chargeback_value  = $_POST['chargeback_value'] ?? null;

                $updateData = [
                    "dept_approval" => 1,
                    "dept_approved_at" => date("Y-m-d H:i:s"),
                    "dept_approved_by" => $_SESSION['user_id'],
                    "chargeback"  => ($chargeback_needed == 1 ? $chargeback_value : null)
                ];
                break;

            case "finance_approve":
                $updateData = [
                    "finance_approval" => 1,
                    "finance_approved_at" => date("Y-m-d H:i:s"),
                    "finance_approved_by" => $_SESSION['user_id']
                ];
                break;

            case "management_approve":
                $updateData = [
                    "management_approval" => 1,
                    "management_approved_at" => date("Y-m-d H:i:s"),
                    "management_approved_by" => $_SESSION['user_id']
                ];
                break;

            case "paid_approve":
                $cash_collector = $_POST['cash_collector'] ?? null;
                
                // Handle file uploads for payment stage
                $upload_dir = __DIR__ . "/../../public/uploads/payments/";
                if (!file_exists($upload_dir)) {
                    mkdir($upload_dir, 0777, true);
                }
                
                $updateData = [
                    "paid_approval" => 1,
                    "paid_approved_at" => date("Y-m-d H:i:s"),
                    "paid_approved_by" => $_SESSION['user_id'],
                    "cash_collector" => $cash_collector
                ];
                
                // Handle file3 and file4 uploads
                if (!empty($_FILES['file3']['name'])) {
                    $file3Tmp  = $_FILES['file3']['tmp_name'];
                    $file3Name = time() . "_3_" . basename($_FILES['file3']['name']);
                    $file3Dest = $upload_dir . $file3Name;
                    if (move_uploaded_file($file3Tmp, $file3Dest)) {
                        $updateData['file3_path'] = "uploads/payments/" . $file3Name;
                    }
                }
                
                if (!empty($_FILES['file4']['name'])) {
                    $file4Tmp  = $_FILES['file4']['tmp_name'];
                    $file4Name = time() . "_4_" . basename($_FILES['file4']['name']);
                    $file4Dest = $upload_dir . $file4Name;
                    if (move_uploaded_file($file4Tmp, $file4Dest)) {
                        $updateData['file4_path'] = "uploads/payments/" . $file4Name;
                    }
                }
                break;

            default:
                echo json_encode(["success" => false, "message" => "Invalid Action"]);
                exit;
        }

        $result = $db->updateData("payment_requests", $updateData, $where);

        if ($result) {
            echo json_encode(["success" => true]);
        } else {
            echo json_encode(["success" => false, "message" => "DB update failed"]);
        }
        exit;
    }

    /* -----------------------------------------------------------
       REJECT REQUEST
    -------------------------------------------------------------*/
    public function reject_request() 
    {
        header('Content-Type: application/json');
        
        $db = new Database();

        $id          = $_POST['id'] ?? null;
        $rejectType  = $_POST['reject_type'] ?? null;
        $reason      = $_POST['reason'] ?? null;

        if (!$id || !$rejectType || !$reason) {
            echo json_encode(["success" => false, "message" => "Missing required data"]);
            exit;
        }

        $updateData = [];

        if ($rejectType == "dept") {
            $updateData = [
                "dept_approval" => -1,
                "dept_approved_at" => date("Y-m-d H:i:s"),
                "dept_approved_by" => $_SESSION['user_id'],
                "dept_notes" => $reason,
            ];
        } 
        else if ($rejectType == "finance") {
            $updateData = [
                "finance_approval" => -1,
                "finance_approved_at" => date("Y-m-d H:i:s"),
                "finance_approved_by" => $_SESSION['user_id'],
                "finance_notes" => $reason,
            ];
        } 
        else if ($rejectType == "management") {
            $updateData = [
                "management_approval" => -1,
                "management_approved_at" => date("Y-m-d H:i:s"),
                "management_approved_by" => $_SESSION['user_id'],
                "management_notes" => $reason,
            ];
        }
        else if ($rejectType == "paid") {
            $updateData = [
                "paid_approval" => -1,
                "paid_approved_at" => date("Y-m-d H:i:s"),
                "paid_approved_by" => $_SESSION['user_id'],
                "paid_notes" => $reason,
            ];
        }

        $updated = $db->updateData("payment_requests", $updateData, ["id" => $id]);

        if ($updated) {
            echo json_encode(["success" => true]);
        } else {
            echo json_encode(["success" => false, "message" => "Failed to update"]);
        }
        exit;
    }

    /* -----------------------------------------------------------
       GET MCA REFS BY CLIENT
    -------------------------------------------------------------*/
    public function get_mca_refs_by_client()
{
    header('Content-Type: application/json');

    $client_id  = $_POST['client_id'] ?? null;
    $paymentfor = isset($_POST['paymentfor']) ? (int)$_POST['paymentfor'] : null;

    if (!isset($_POST['client_id']) || !isset($_POST['paymentfor'])) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid request',
            'data' => []
        ]);
        exit;
    }

    $db = new Database();

    // Table & column mapping
    $mapping = [
        0 => ['table' => 'imports_t', 'client_col' => 'subscriber_id',     'ref_col' => 'mca_ref'],
        1 => ['table' => 'exports_t', 'client_col' => 'subscriber_id',     'ref_col' => 'mca_ref'],
        2 => ['table' => 'locals_t',  'client_col' => 'clients_id', 'ref_col' => 'lt_reference']
    ];

    if (!isset($mapping[$paymentfor])) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid payment_for value',
            'data' => []
        ]);
        exit;
    }

    $table     = $mapping[$paymentfor]['table'];
    $clientCol = $mapping[$paymentfor]['client_col'];
    $refCol    = $mapping[$paymentfor]['ref_col'];

    $sql  = "SELECT id, $refCol AS ref FROM $table WHERE $clientCol = :client_id";
    $rows = $db->customQuery($sql, [':client_id' => $client_id]);

    if (!$rows) {
        echo json_encode([
            'success' => false,
            'message' => 'No rows found for client ID ' . $client_id,
            'data' => []
        ]);
        exit;
    }

    $finalList = [];

    foreach ($rows as $row) {

        if (!$row['ref']) continue;

        $refValue = trim($row['ref']);
        if ($refValue === '') continue;

        // Try json_decode
        $decoded = json_decode($refValue, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            // Not JSON → treat as plain string
            $finalList[] = [
                'payment_id' => $row['id'],
                'mca_ref'    => $refValue
            ];
            continue;
        }

        // JSON list (ex: [{mca_ref: "X"}])
        foreach ($decoded as $item) {
            $finalList[] = [
                'payment_id' => $row['id'],
                'mca_ref'    => $item['mca_ref'] ?? ''
            ];
        }
    }

    echo json_encode([
        'success' => true,
        'data'    => $finalList
    ]);
    exit;
}
/* -----------------------------------------------------------
       EXPORT TO EXCEL
    -------------------------------------------------------------*/
    public function export_to_excel()
    {
        header('Content-Type: application/json');
        
        $db = new Database();
        
        $sql = "SELECT pr.*, d.department_name, c.short_name AS client_name,
                       cu.currency_short_name, of.main_location_name, ex.expense_type_name
                FROM payment_requests pr
                LEFT JOIN department_master_t d ON d.id = pr.department
                LEFT JOIN clients_t c ON c.id = pr.client_id
                LEFT JOIN currency_master_t cu ON cu.id = pr.currency
                LEFT JOIN main_office_master_t of ON of.id = pr.location_id
                LEFT JOIN expense_type_master_t ex ON ex.id = pr.expense_type
                ORDER BY pr.id DESC";
        
        $data = $db->customQuery($sql);
        
        echo json_encode([
            'success' => true,
            'data' => $data
        ]);
        exit;
    }
}